-- pack.sql (MySQL/MariaDB compatible) - Calificación Crediticia
-- Ejecuta este script con un usuario que tenga permisos ALTER/CREATE VIEW
-- Nota: las ALTER TABLE se hacen columna por columna (sin IF NOT EXISTS). Si ya existían,
-- tu herramienta puede mostrar un error de "Duplicate column". Es seguro ignorarlo.
-- Si prefieres verificar primero: SHOW COLUMNS FROM llx_societe_extrafields LIKE 'cc_category';

-- 1) Asegura filas en extrafields
INSERT INTO llx_societe_extrafields (fk_object)
SELECT s.rowid FROM llx_societe s
LEFT JOIN llx_societe_extrafields se ON se.fk_object = s.rowid
WHERE se.fk_object IS NULL;

-- 2) Agregar columnas requeridas (idempotencia manual: ejecutar una a una si es necesario)
ALTER TABLE llx_societe_extrafields ADD COLUMN cc_category VARCHAR(20) DEFAULT NULL;
ALTER TABLE llx_societe_extrafields ADD COLUMN cc_score DECIMAL(5,2) DEFAULT NULL;
ALTER TABLE llx_societe_extrafields ADD COLUMN cc_updated_at DATETIME DEFAULT NULL;
ALTER TABLE llx_societe_extrafields ADD COLUMN cc_veto_reason VARCHAR(255) DEFAULT NULL;
ALTER TABLE llx_societe_extrafields ADD COLUMN cc_overdue_ratio DECIMAL(10,4) DEFAULT NULL;
ALTER TABLE llx_societe_extrafields ADD COLUMN cc_dso DECIMAL(10,2) DEFAULT NULL;
ALTER TABLE llx_societe_extrafields ADD COLUMN cc_cash_ratio DECIMAL(10,4) DEFAULT NULL;
ALTER TABLE llx_societe_extrafields ADD COLUMN options_cc_cheque_devuelto TINYINT(1) DEFAULT 0;

-- 3) Tabla de snapshots (histórico)
CREATE TABLE IF NOT EXISTS llx_customer_cc_snapshot (
  rowid INT AUTO_INCREMENT PRIMARY KEY,
  fk_soc INT NOT NULL,
  period_label VARCHAR(20) NOT NULL,
  cc_score DECIMAL(5,2),
  cc_category VARCHAR(20),
  cc_veto_reason VARCHAR(255),
  dso DECIMAL(10,2),
  cash_ratio DECIMAL(10,4),
  overdue_ratio DECIMAL(10,4),
  max_days_overdue INT,
  returns_ratio DECIMAL(10,4),
  margin_rate DECIMAL(10,4),
  monthly_sales DECIMAL(24,8),
  active_months_12m INT,
  bounced_checks_365d INT,
  datec DATETIME DEFAULT CURRENT_TIMESTAMP,
  KEY idx_soc (fk_soc),
  KEY idx_datec (datec)
) ENGINE=InnoDB;

-- 4) Vistas (todas se recrean; usa DROP VIEW IF EXISTS antes por compatibilidad)

-- 4.1 Pagos 90d: DSO y %Contado
DROP VIEW IF EXISTS llx_vw_cc_payments_90d;
CREATE VIEW llx_vw_cc_payments_90d AS
SELECT
  f.fk_soc,
  SUM(pf.amount * DATEDIFF(p.datep, f.datef)) / NULLIF(SUM(pf.amount),0) AS dso_days,
  SUM(CASE WHEN DATEDIFF(p.datep, f.datef) <= 1 THEN pf.amount ELSE 0 END) / NULLIF(SUM(pf.amount),0) AS cash_ratio
FROM llx_facture f
JOIN llx_paiement_facture pf ON pf.fk_facture = f.rowid
JOIN llx_paiement p ON p.rowid = pf.fk_paiement
WHERE f.fk_statut IN (1,2)
  AND f.type IN (0,1,2,3)
  AND p.datep >= DATE_SUB(CURDATE(), INTERVAL 90 DAY)
GROUP BY f.fk_soc;

-- 4.2 Cartera y vencidos
DROP VIEW IF EXISTS llx_vw_cc_overdue;
CREATE VIEW llx_vw_cc_overdue AS
SELECT
  f.fk_soc,
  SUM(f.total_ttc - IFNULL(paid.amount,0)) AS open_balance,
  SUM(CASE WHEN f.date_lim_reglement < CURDATE() THEN (f.total_ttc - IFNULL(paid.amount,0)) ELSE 0 END) AS overdue_balance,
  MAX(GREATEST(0, DATEDIFF(CURDATE(), f.date_lim_reglement))) AS max_days_overdue
FROM llx_facture f
LEFT JOIN (
  SELECT fk_facture, SUM(amount) AS amount
  FROM llx_paiement_facture
  GROUP BY fk_facture
) paid ON paid.fk_facture = f.rowid
WHERE f.fk_statut IN (1)
  AND (f.paye = 0 OR (f.total_ttc - IFNULL(paid.amount,0)) > 0)
GROUP BY f.fk_soc;

-- 4.3 Ventas y margen 180d
DROP VIEW IF EXISTS llx_vw_cc_sales_margin_180d;
CREATE VIEW llx_vw_cc_sales_margin_180d AS
SELECT
  f.fk_soc,
  SUM(CASE WHEN f.type IN (0,1) THEN f.total_ht ELSE 0 END) AS total_ht_sales_180d,
  COUNT(DISTINCT DATE_FORMAT(f.datef,'%Y-%m')) AS active_months_180d,
  (SUM(CASE WHEN f.type IN (0,1) THEN (fd.total_ht - (COALESCE(fd.buy_price_ht,0) * fd.qty)) ELSE 0 END)
   / NULLIF(SUM(CASE WHEN f.type IN (0,1) THEN fd.total_ht ELSE 0 END),0)) AS margin_rate
FROM llx_facture f
JOIN llx_facturedet fd ON fd.fk_facture = f.rowid
WHERE f.fk_statut IN (1,2)
  AND f.datef >= DATE_SUB(CURDATE(), INTERVAL 180 DAY)
GROUP BY f.fk_soc;

-- 4.4 Devoluciones 180d
DROP VIEW IF EXISTS llx_vw_cc_returns_180d;
CREATE VIEW llx_vw_cc_returns_180d AS
SELECT
  f.fk_soc,
  SUM(CASE WHEN f.type IN (2) THEN -f.total_ht ELSE 0 END) AS credit_ht_180d,
  SUM(CASE WHEN f.type IN (0,1) THEN f.total_ht ELSE 0 END) AS sales_ht_180d,
  (SUM(CASE WHEN f.type IN (2) THEN -f.total_ht ELSE 0 END) / NULLIF(SUM(CASE WHEN f.type IN (0,1) THEN f.total_ht ELSE 0 END),0)) AS returns_ratio
FROM llx_facture f
WHERE f.fk_statut IN (1,2)
  AND f.datef >= DATE_SUB(CURDATE(), INTERVAL 180 DAY)
GROUP BY f.fk_soc;

-- 4.5 Aliases sin prefijo llx_ (opcionales)
DROP VIEW IF EXISTS vw_cc_payments_90d;
CREATE VIEW vw_cc_payments_90d AS SELECT * FROM llx_vw_cc_payments_90d;

DROP VIEW IF EXISTS vw_cc_overdue;
CREATE VIEW vw_cc_overdue AS SELECT * FROM llx_vw_cc_overdue;

DROP VIEW IF EXISTS vw_cc_sales_margin_180d;
CREATE VIEW vw_cc_sales_margin_180d AS SELECT * FROM llx_vw_cc_sales_margin_180d;

DROP VIEW IF EXISTS vw_cc_returns_180d;
CREATE VIEW vw_cc_returns_180d AS SELECT * FROM llx_vw_cc_returns_180d;
