<?php
/**
 * Backfill masivo Calificación Crediticia (CLI y desde Setup)
 * Uso CLI: php cc_backfill.php [entity]
 */

// ======= Bootstrap seguro para CLI =======
if (PHP_SAPI === 'cli') {
    // Evita exigir login/CSRF/menús en CLI
    define('NOLOGIN', 1);
    define('NOCSRFCHECK', 1);
    define('NOREQUIREMENU', 1);
    define('NOREQUIREHTML', 1);
    define('NOTOKENRENEWAL', 1);
}

// Cargar main.inc.php con ruta robusta (sube 3 niveles desde /custom/.../scripts)
$main = dirname(__DIR__, 3) . '/main.inc.php';
if (!is_file($main)) {
    // fallback por si tu árbol difiere
    $main = __DIR__ . '/../../../../main.inc.php';
}
// Tercer intento: ruta absoluta de tu servidor en cPanel
if (!is_file($main)) {
    $main = '/home/aliment5/public_html/sistema/htdocs/main.inc.php';
}
if (!is_file($main)) {
    if (PHP_SAPI === 'cli') fwrite(STDERR, "No se encontró main.inc.php. Ajusta la ruta en cc_backfill.php\n");
    else print "No se encontró main.inc.php. Ajusta la ruta en cc_backfill.php";
    exit(1);
}
require_once $main;

global $db, $conf, $langs;

// ======= Contexto usuario/admin para CLI =======
if (PHP_SAPI === 'cli') {
    // Entidad (por defecto 1; puedes pasarla como primer argumento)
    $entity = isset($argv[1]) ? (int)$argv[1] : 1;
    $conf->entity = $entity;

    // Usuario admin técnico
    dol_include_once('/user/class/user.class.php');
    $user = new User($db);
    $user->fetch(1);           // <-- ID 1 suele ser admin; cambia si tu admin tiene otro ID
    $user->admin = 1;          // fuerza privilegios admin en memoria
    // Si usas multicompany y tu admin usa otra entidad, ajústalo o crea un usuario válido para esa entidad.
} else {
    // En web el $user ya viene de la sesión; si no es admin, bloqueará en Setup (no aquí)
}

// ======= Helpers =======
function cc_sql_one($sql)
{
    global $db;
    $res = $db->query($sql);
    if (!$res) return null;
    $o = $db->fetch_object($res);
    if (!$o) return null;
    $props = array_keys(get_object_vars($o));
    return $props ? $o->{$props[0]} : null;
}

function cc_log($msg, $verbose=true)
{
    if ($verbose) print rtrim($msg).PHP_EOL;
}

// ======= Lógica principal =======
function cc_backfill_run($entity = 1, $verbose = true)
{
    global $db, $conf;

    // Ponderaciones (usa consts si existen, si no defaults)
    $W = [
        'DSO'     => (float)($conf->global->CC_W_DSO     ?? 0.25),
        'CASH'    => (float)($conf->global->CC_W_CASH    ?? 0.20),
        'OVERDUE' => (float)($conf->global->CC_W_OVERDUE ?? 0.30),
        'MARGIN'  => (float)($conf->global->CC_W_MARGIN  ?? 0.00), // pon 0 si aún no usas margen
    ];
    $autoRenorm = (int)($conf->global->CC_AUTO_RENORM ?? 1);

    // Umbrales/categorías
    $THR = [
        'PREMIUM'    => (float)($conf->global->CC_THR_PREMIUM    ?? 85),
        'ESTANDAR'   => (float)($conf->global->CC_THR_ESTANDAR   ?? 70),
        'ENATENCION' => (float)($conf->global->CC_THR_ENATENCION ?? 55),
    ];

    // Vetos
    $VETO_MAXOVERDUEDAYS = (int)($conf->global->CC_VETO_MAXOVERDUEDAYS ?? 45);

    cc_log("== Backfill CC (entity={$entity}) ==", $verbose);

    // Selección de clientes
    $sql = "SELECT s.rowid AS socid, s.nom AS name
            FROM ".MAIN_DB_PREFIX."societe s
            WHERE s.client IN (1,2,3)";
    $res = $db->query($sql);
    if (!$res) {
        cc_log("ERROR SQL: ".$db->lasterror(), true);
        return 0;
    }

    $count = 0;
    while ($o = $db->fetch_object($res)) {
        $socid = (int)$o->socid;

        // Métricas desde las vistas
        $dso        = (float)(cc_sql_one("SELECT dso_days FROM vw_cc_payments_90d WHERE fk_soc=".$socid) ?? 0);
        $cash_ratio = (float)(cc_sql_one("SELECT cash_ratio FROM vw_cc_payments_90d WHERE fk_soc=".$socid) ?? 0);
        $over_ratio = (float)(cc_sql_one("SELECT overdue_ratio FROM vw_cc_overdue WHERE fk_soc=".$socid) ?? 0);
        $max_over   = (int)  (cc_sql_one("SELECT max_days_overdue FROM vw_cc_overdue WHERE fk_soc=".$socid) ?? 0);
        $margin_rt  = (float)(cc_sql_one("SELECT margin_rate FROM vw_cc_sales_margin_180d WHERE fk_soc=".$socid) ?? 0);

        // Escoring por métrica (0..1)
        $s_dso = ($dso <= 5) ? 1.0 : (($dso <= 10) ? 0.6 : (($dso <= 15) ? 0.4 : 0.1));
        $s_cash = max(0.0, min(1.0, $cash_ratio));
        $base_over = ($over_ratio <= 0 ? 1.0 : (($over_ratio <= 0.10) ? 0.7 : (($over_ratio <= 0.25) ? 0.4 : 0.1)));
        if ($max_over > 60) $base_over -= 0.1;
        $s_over = max(0.0, $base_over);
        $s_marg = ($margin_rt <= 0.05 ? 0.2 : ($margin_rt <= 0.10 ? 0.4 : ($margin_rt <= 0.20 ? 0.7 : 1.0)));

        // Score ponderado 0..100
        $sumW = 0.0; $sumWX = 0.0;
        foreach ([['DSO',$s_dso],['CASH',$s_cash],['OVERDUE',$s_over],['MARGIN',$s_marg]] as [$k,$v]) {
            $w = $W[$k];
            if ($w > 0) { $sumW += $w; $sumWX += $w * $v; }
        }
        $score01 = ($sumW>0) ? ($autoRenorm ? ($sumWX/$sumW) : $sumWX) : 0.0;
        $score = max(0.0, min(100.0, $score01 * 100.0));

        // Vetos
        $veto = null;
        if ($max_over >= $VETO_MAXOVERDUEDAYS && $over_ratio > 0.10) $veto = 'OverdueVeto';
        $bounced = (int)(cc_sql_one("SELECT options_cc_cheque_devuelto FROM ".MAIN_DB_PREFIX."societe_extrafields WHERE fk_object=".$socid) ?? 0);
        if ($bounced > 0) $veto = 'BouncedCheck';

        // Categoría final (4 niveles)
        if ($veto) $cat = 'AltoRiesgo';
        else if ($score >= $THR['PREMIUM']) $cat = 'Premium';
        else if ($score >= $THR['ESTANDAR']) $cat = 'Estandar';
        else if ($score >= $THR['ENATENCION']) $cat = 'EnAtencion';
        else $cat = 'AltoRiesgo';

        // Asegura la fila extrafield
        $db->query("INSERT IGNORE INTO ".MAIN_DB_PREFIX."societe_extrafields (fk_object) VALUES (".$socid.")");

        // Actualiza extrafields
        $upd = "UPDATE ".MAIN_DB_PREFIX."societe_extrafields SET
                    cc_score=".round($score,2).",
                    cc_category='".$db->escape($cat)."',
                    cc_updated_at=NOW(),
                    cc_veto_reason=".($veto ? "'".$db->escape($veto)."'" : "NULL").",
                    cc_overdue_ratio=".$over_ratio.",
                    cc_dso=".$dso.",
                    cc_cash_ratio=".$cash_ratio."
                WHERE fk_object=".$socid;
        if (!$db->query($upd)) {
            cc_log("ERR socid={$socid} -> ".$db->lasterror(), true);
        } else {
            $count++;
            if ($verbose && ($count % 50) === 0) cc_log("Procesados: {$count}", true);
        }
    }

    cc_log("Backfill completado. Total actualizados: {$count}", $verbose);
    return $count;
}

// ======= Si se ejecuta directamente por CLI =======
if (PHP_SAPI === 'cli' && realpath($_SERVER['argv'][0]) === __FILE__) {
    $entity = isset($argv[1]) ? (int)$argv[1] : 1;
    cc_backfill_run($entity, true);
}
