<?php
class CCService
{
    public static function computeForSoc($db, $socid, $conf)
    {
        $m = self::loadMetrics($db, $socid);

        $w = array(
            'dso'     => floatval($conf->global->CC_W_DSO ?: 0.25),
            'cash'    => floatval($conf->global->CC_W_CASH ?: 0.20),
            'overdue' => floatval($conf->global->CC_W_OVERDUE ?: 0.15),
            'margin'  => floatval($conf->global->CC_W_MARGIN ?: 0.00),
            'sales'   => floatval($conf->global->CC_W_SALES ?: 0.05),
            'tenure'  => floatval($conf->global->CC_W_TENURE ?: 0.10),
            'returns' => floatval($conf->global->CC_W_RETURNS ?: 0.05),
            'events'  => floatval($conf->global->CC_W_EVENTS ?: 0.00),
        );
        $auto = !empty($conf->global->CC_AUTO_RENORM);

        $sc = array(
            'dso'     => self::score_dso($m['dso_days']),
            'cash'    => self::score_cash($m['cash_ratio']),
            'overdue' => self::score_overdue($m['overdue_ratio'], $m['max_days_overdue']),
            'margin'  => self::score_margin($m['margin_rate']),
            'sales'   => self::score_sales($m['monthly_sales']),
            'tenure'  => self::score_tenure($m['active_months_12m']),
            'returns' => self::score_returns($m['returns_ratio']),
            'events'  => self::score_events($m['bounced_checks_365d']),
        );

        $sumW=0.0; $sumWX=0.0;
        foreach ($w as $k=>$wk) if ($wk>0) { $sumW+=$wk; $sumWX += $wk*$sc[$k]; }
        $score01 = ($sumW>0) ? ($auto ? ($sumWX/$sumW) : $sumWX) : 0;
        $score = max(0, min(100, $score01*100));

        list($category, $veto) = self::categorize4($m, $score, $conf);
        self::saveExtrafields($db, $socid, $score, $category, $veto, $m);
        self::snapshot($db, $socid, $score, $category, $veto, $m);
        return array($score, $category, $veto);
    }

    public static function loadMetrics($db, $socid)
    {
        $m = array(
            'dso_days'=>0,'cash_ratio'=>0,'overdue_ratio'=>0,'max_days_overdue'=>0,
            'margin_rate'=>0,'monthly_sales'=>0,'active_months_12m'=>0,
            'returns_ratio'=>0,'bounced_checks_365d'=>0
        );

        $sql = "SELECT dso_days, cash_ratio FROM ".MAIN_DB_PREFIX."vw_cc_payments_90d WHERE fk_soc=".(int)$socid;
        $res = $db->query($sql); if ($res && ($o=$db->fetch_object($res))) { $m['dso_days']=(float)$o->dso_days; $m['cash_ratio']=(float)$o->cash_ratio; }

        $sql = "SELECT open_balance, overdue_balance, max_days_overdue FROM ".MAIN_DB_PREFIX."vw_cc_overdue WHERE fk_soc=".(int)$socid;
        $res = $db->query($sql); if ($res && ($o=$db->fetch_object($res))) {
            $open = (float)$o->open_balance; $over=(float)$o->overdue_balance;
            $m['overdue_ratio'] = ($open>0? $over/$open : 0); $m['max_days_overdue']=(int)$o->max_days_overdue;
        }

        $sql = "SELECT total_ht_sales_180d, active_months_180d, margin_rate FROM ".MAIN_DB_PREFIX."vw_cc_sales_margin_180d WHERE fk_soc=".(int)$socid;
        $res = $db->query($sql); if ($res && ($o=$db->fetch_object($res))) {
            $m['monthly_sales'] = ((float)$o->total_ht_sales_180d)/6.0;
            $m['active_months_12m'] = (int)$o->active_months_180d;
            $m['margin_rate'] = (float)$o->margin_rate;
        }

        $sql = "SELECT returns_ratio FROM ".MAIN_DB_PREFIX."vw_cc_returns_180d WHERE fk_soc=".(int)$socid;
        $res = $db->query($sql); if ($res && ($o=$db->fetch_object($res))) $m['returns_ratio']=(float)$o->returns_ratio;

        $m['bounced_checks_365d'] = self::bouncedChecksLast365($db, $socid);
        return $m;
    }

    public static function bouncedChecksLast365($db, $socid)
    {
        global $conf;
        $mode = !empty($conf->global->CC_EVENTS_MODE) ? $conf->global->CC_EVENTS_MODE : 'MANUAL';
        $n = 0;

        if ($mode=='MANUAL' || $mode=='HYBRID') {
            $sql = "SELECT se.options_cc_cheque_devuelto AS flag FROM ".MAIN_DB_PREFIX."societe_extrafields se WHERE se.fk_object=".(int)$socid;
            $res = $db->query($sql); if ($res && ($o=$db->fetch_object($res))) { if (!empty($o->flag)) $n = max($n,1); }
        }

        if ($mode=='AUTO' || $mode=='HYBRID') {
            $labels = !empty($conf->global->CC_EVENTS_LABELS) ? explode(',', $conf->global->CC_EVENTS_LABELS) : array('DEVUELTO','RECHAZADO');
            $like = array();
            foreach ($labels as $L){ $L = trim($L); if ($L!=='') $like[] = "b.label LIKE '%".$db->escape($L)."%'"; }
            $cond = count($like)? (' AND ('.implode(' OR ', $like).') ') : '';

            $sql = "SELECT COUNT(*) n
                    FROM ".MAIN_DB_PREFIX."bank b
                    JOIN ".MAIN_DB_PREFIX."paiement p ON p.fk_bank = b.rowid
                    JOIN ".MAIN_DB_PREFIX."paiement_facture pf ON pf.fk_paiement = p.rowid
                    JOIN ".MAIN_DB_PREFIX."facture f ON f.rowid = pf.fk_facture
                    WHERE f.fk_soc=".(int)$socid."
                      AND b.dateo >= DATE_SUB(CURDATE(), INTERVAL 365 DAY)".$cond;
            $res = $db->query($sql); $o=$res?$db->fetch_object($res):null; $n = max($n, ($o?intval($o->n):0));
        }
        return $n;
    }

    private static function interp($x,$x0,$y0,$x1,$y1){ if($x<=$x0)return $y0; if($x>=$x1)return $y1; return $y0+($y1-$y0)*($x-$x0)/($x1-$x0); }
    private static function score_dso($d){ if($d<=0)return 1; if($d<=5)return 1; if($d<=7)return 0.85; if($d<=10)return self::interp($d,7,0.85,10,0.60); if($d<=15)return self::interp($d,10,0.60,15,0.40); return 0.10; }
    private static function score_cash($r){ if($r<=0)return 0; if($r<=0.5)return self::interp($r,0,0,0.5,0.70); if($r<=0.7)return self::interp($r,0.5,0.70,0.7,0.90); if($r<=0.9)return self::interp($r,0.7,0.90,0.9,1.00); return 1.00; }
    private static function score_overdue($ratio,$maxdays){ $base=1.0; if($ratio<=0)$base=1.0; else if($ratio<=0.10)$base=0.70; else if($ratio<=0.25)$base=0.40; else $base=0.10; if($maxdays>3 && $maxdays<=7) $base-=0.1;else if($maxdays>7 && $maxdays<=25) $base-=0.2;else if($maxdays>25) $base-=0.4; return max(0,$base); }
    private static function score_margin($m){ if($m<=0.05)return 0.20; if($m<=0.10)return 0.40; if($m<=0.20)return 0.70; return 1.00; }
    private static function score_sales($m){ if($m<=0)return 0; return max(0,min(1,1-1/(1+$m/1000.0))); }
    private static function score_tenure($months){ if($months>=12)return 1.0; if($months>=6)return 0.70; if($months>=3)return 0.40; return 0.20; }
    private static function score_returns($r){ if($r<=0)return 1.0; if($r<=0.05)return 0.80; if($r<=0.10)return 0.50; return 0.20; }
    private static function score_events($n){ $pen=min(0.60,0.30*max(0,$n)); return max(0,1.0-$pen); }

    public static function categorize4($m, $score, $conf)
    {
        $thrPremium  = floatval($conf->global->CC_THR_PREMIUM ?: 85);
        $thrEstandar = floatval($conf->global->CC_THR_ESTANDAR ?: 70);
        $thrAtencion = floatval($conf->global->CC_THR_ENATENCION ?: 55);

        $vetoMaxDays    = intval($conf->global->CC_VETO_MAXOVERDUEDAYS ?: 45);
        $vetoBounced365 = intval($conf->global->CC_VETO_BOUNCED_365 ?: 2);
        $vetoBounced180 = intval($conf->global->CC_VETO_BOUNCED_180 ?: 1);

        $veto = '';
        if ($m['bounced_checks_365d'] >= $vetoBounced365) $veto = 'BouncedChecks365';
        if ($m['bounced_checks_365d'] >= $vetoBounced180) $veto = 'BouncedChecks180';
        if ($m['max_days_overdue'] >= $vetoMaxDays && $m['overdue_ratio'] > 0.10) $veto = 'OverdueVeto';

        if ($veto) return array('AltoRiesgo', $veto);
        if ($score >= $thrPremium) return array('Premium','');
        if ($score >= $thrEstandar) return array('Estandar','');
        if ($score >= $thrAtencion) return array('EnAtencion','');
        return array('AltoRiesgo','LowScore');
    }

    public static function saveExtrafields($db,$socid,$score,$cat,$veto,$m)
    {
        $socid=(int)$socid;
        $db->query("INSERT IGNORE INTO ".MAIN_DB_PREFIX."societe_extrafields (fk_object) VALUES (".$socid.")");
        $sql="UPDATE ".MAIN_DB_PREFIX."societe_extrafields SET ".
            "cc_score=".((float)$score).",".
            "cc_category='".$db->escape($cat)."',".
            "cc_veto_reason=".($veto?"'".$db->escape($veto)."'":"NULL").",".
            "cc_dso=".((float)$m['dso_days']).",".
            "cc_cash_ratio=".((float)$m['cash_ratio']).",".
            "cc_overdue_ratio=".((float)$m['overdue_ratio']).",".
            "cc_updated_at=NOW() ".
            "WHERE fk_object=".$socid;
        $db->query($sql);
    }

    public static function snapshot($db,$socid,$score,$cat,$veto,$m)
    {
        $sql="INSERT INTO ".MAIN_DB_PREFIX."customer_cc_snapshot
            (fk_soc, period_label, cc_score, cc_category, cc_veto_reason, dso, cash_ratio, overdue_ratio, max_days_overdue, returns_ratio, margin_rate, monthly_sales, active_months_12m, bounced_checks_365d, datec)
            VALUES (".(int)$socid.", 'D', ".((float)$score).", '".$db->escape($cat)."', ".($veto?"'".$db->escape($veto)."'":"NULL").",
            ".((float)$m['dso_days']).", ".((float)$m['cash_ratio']).", ".((float)$m['overdue_ratio']).", ".(int)$m['max_days_overdue'].", ".((float)$m['returns_ratio']).",
            ".((float)$m['margin_rate']).", ".((float)$m['monthly_sales']).", ".(int)$m['active_months_12m'].", ".(int)$m['bounced_checks_365d'].", NOW())";
        $db->query($sql);
    }
}
